<?php
use Carbon\Carbon;

/**
 * Calculate the total paid amount from the transactions.
 *
 * @param  mixed  $transactions  The transactions collection.
 * @return float                 The total paid amount.
 */
if (!function_exists('paid_amount')) {
	function paid_amount($transactions)
	{
		$paid = sum_transaction($transactions, 'cr');

		return $paid;
	}
}

if (!function_exists('due_amount')) {
	function due_amount($transactions)
	{
		include app_path() . '/Helpers/general.php';

		$total = (int) sum_transaction($transactions, 'dr');
		$paid =  (int) sum_transaction($transactions, 'cr');
		
		$due = $total - $paid;
		return round_off($due);
	}
}

/**
 * Truncate a string if it exceeds the specified length.
 *
 * @param  string  $string  The string to truncate.
 * @param  int     $length  The maximum length of the truncated string.
 * @return string           The truncated string.
 */
if (!function_exists('truncate_string')) {
	function truncate_string($string, $length)
	{
		if (strlen($string) > $length) {
			return substr($string, 0, $length) . '...';
		}
		return $string;
	}
}

/**
 * Calculate the total amount of transactions of a specific type.
 *
 * @param  mixed  $transactions  The transactions collection.
 * @param  string $type          The type of transaction ('dr' for debit, 'cr' for credit).
 * @return float                 The total amount of transactions.
 */
if (!function_exists('sum_transaction')) {
	function sum_transaction($transactions, $type)
	{
		include app_path() . '/Helpers/general.php';

		$total = 0;
		foreach ($transactions as $transaction) {
			if ($transaction->dr_cr == $type) {
				$total = $total + $transaction->amount;
			}
		}

		return round_off($total);
	}
}

/**
 * Determine the status of a set of transactions.
 *
 * @param  mixed  $transactions  The transactions collection.
 * @return string                The status badge HTML.
 */
if (!function_exists('transaction_status')) {
	function transaction_status($transactions)
	{
		$total = sum_transaction($transactions, 'dr');
		$paid = sum_transaction($transactions, 'cr');

		if ($paid < $total && $paid != 0) {
			return '<span class="badge badge-warning"">Part Payment</span>';
		} else if ($paid >= $total) {
			return '<span class="badge badge-success"">Completed</span>';
		} else {
			return '<span class="badge badge-primary">Due</span>';
		}
	}
}

if (!function_exists('transaction_type')) {
	function transaction_type($transaction)
	{
		if ($transaction->dr_cr == 'cr') {
			return '<span class="text-success"">' . '+ ' . round_off($transaction->amount) . '</span>';
		} else {
			return '<span class="text-primary"">' . '- ' . round_off($transaction->amount) . '</span>';
		}
	}
}

/**
 * Determine the payment status based on actual and paid amounts.
 *
 * @param  float  $actual  The actual amount.
 * @param  float  $paid    The paid amount.
 * @return string          The payment status badge HTML.
 */
if (!function_exists('payment_status')) {
	function payment_status(float $actual, float $paid)
	{
		if ($paid < $actual && $paid != 0) {
			return '<span class="badge badge-warning"">Part Payment</span>';
		} else if ($paid >= $actual) {
			return '<span class="badge badge-success"">Completed</span>';
		} else {
			return '<span class="badge badge-primary">Due</span>';
		}
	}
}

if (!function_exists('membership_status')) {
	function membership_status($valid_to)
	{
		$valid_to_date = Carbon::parse($valid_to)->startOfDay();

		$today = Carbon::now()->startOfDay();
		$one_week_later = $today->copy()->addWeek();

		if ($valid_to_date->isBefore($today)) {
			return '<span class="badge badge-primary"">Expired</span>';
		} elseif ($valid_to_date->isBetween($today, $one_week_later)) {
			return '<span class="badge badge-warning"">Near Expiry</span>';
		} else {
			return '<span class="badge badge-success"">Valid</span>';
		}
	}
}

/**
 * Get the membership status text based on the validity date.
 *
 * @param  string  $valid_to  The validity date of the membership.
 * @return string             The membership status text.
 */
if (!function_exists('membership_status_text')) {
	function membership_status_text($valid_to)
	{
		$valid_to_date = Carbon::parse($valid_to)->startOfDay();

		$today = Carbon::now()->startOfDay();
		$one_week_later = $today->copy()->addWeek();

		if ($valid_to_date->isBefore($today)) {
			return 'Expired';
		} elseif ($valid_to_date->isBetween($today, $one_week_later)) {
			return 'Near Expiry';
		} else {
			return 'Valid';
		}
	}
}

/**
 * Get the start and end dates of the month in a human-readable format.
 *
 * @param  string|\Carbon\Carbon  $date  The date to determine the month's start and end dates.
 * @return array                        An array containing the start and end dates of the month.
 */
if (!function_exists('month_dates')) {
	function month_dates($date)
	{
		$date = Carbon::parse($date);
		$date1 = $date->startOfMonth()->format('jS M Y');
		$date2 = $date->endOfMonth()->format('jS M Y');

		return [
			'date1' => $date1,
			'date2' => $date2,
		];
	}
}

/**
 * Generate a human-readable date range for attendance within a month.
 *
 * @param  string|\Carbon\Carbon  $date  The date to determine the month's start and end dates.
 * @return string                         A formatted string indicating the attendance period within the month.
 */
if (!function_exists('date_range')) {
	function date_range($date)
	{
		$range = month_dates($date);
		return [ 'date1' => $range['date1'], 'date2' => $range['date2'] ];
	}
}

/**
 * Modify the day of a given date and return the modified date.
 *
 * @param  string|\Carbon\Carbon  $date  The date to modify.
 * @param  int                    $day   The day to set.
 * @return string                         The modified date in 'Y-m-d' format.
 */
if (!function_exists('modify_day')) {
	function modify_day($date, $day)
	{
		$date = Carbon::parse($date);

		$date->day($day);

		return $date->format('Y-m-d');
	}
}

/**
 * Check the attendance status for a given day.
 *
 * @param  \Illuminate\Support\Collection  $attendances  The collection of attendances.
 * @param  int                              $day          The day to check the status for.
 * @return string|null                                    The attendance status ('present', 'absent', or null if not found).
 */
if (!function_exists('checkStatus')) {
	function checkStatus($attendances, $day)
	{
		$date = session('attendance_date');
		$modified_date = modify_day($date, $day);

		foreach ($attendances as $attendance) {
			$attendance_date = Carbon::parse($attendance->attendance_date)->format('Y-m-d');
			if ($attendance_date == $modified_date) {
				if ($attendance->status == 'present') {
					return 'present';
				} else if ($attendance->status == 'absent') {
					return 'absent';
				} else {
					// NOTHING
				}
			}
		}
	}
}

/**
 * Count the number of 'present' attendances in the provided collection.
 *
 * @param  \Illuminate\Support\Collection  $attendances  The collection of attendances.
 * @return int                                           The count of 'present' attendances.
 */
if (!function_exists('presentCount')) {
	function presentCount($attendances)
	{
		return $attendances->filter(function ($attendance) {
			return $attendance->status === 'present';
		})->count();
	}
}

/**
 * Count the number of 'absent' attendances in the provided collection.
 *
 * @param  \Illuminate\Support\Collection  $attendances  The collection of attendances.
 * @return int                                           The count of 'absent' attendances.
 */
if (!function_exists('absentCount')) {
	function absentCount($attendances)
	{
		return $attendances->filter(function ($attendance) {
			return $attendance->status === 'absent';
		})->count();
	}
}

/**
 * Subtract one month from the given date.
 *
 * @param  string|\Carbon\Carbon  $date  The date to subtract one month from.
 * @return \Carbon\Carbon                 The date with one month subtracted.
 */
if (!function_exists('sub_month')) {
	function sub_month($date)
	{
		$date = Carbon::parse($date);

		$previous_month = $date->copy()->subMonth();

		return $previous_month;
	}
}

/**
 * Add one month to the given date.
 *
 * @param  string|\Carbon\Carbon  $date  The date to add one month to.
 * @return \Carbon\Carbon                 The date with one month added.
 */
if (!function_exists('add_month')) {
	function add_month($date)
	{
		$date = Carbon::parse($date);

		$next_month = $date->copy()->addMonth();

		return $next_month;
	}
}

/**
 * Get the total number of days in the month of the given date.
 *
 * @param  string|\Carbon\Carbon  $date  The date for which to get the total days in the month.
 * @return int                           The total number of days in the month.
 */
if (!function_exists('total_days')) {
	function total_days($date)
	{
		$date = Carbon::parse($date);

		$total = $date->daysInMonth;

		return $total;
	}
}

/**
 * Get a badge HTML element based on the status.
 *
 * @param  string  $status  The status value ('active' or 'disabled').
 * @return string          The HTML badge element representing the status.
 */
if (!function_exists('status_badge')) {
	function status_badge($status)
	{
		if ($status == 'active') {
			return '<span class="badge badge-success"">Active</span>';
		} else {
			return '<span class="badge badge-primary">Disabled</span>';
		}
	}
}

/**
 * Get the URL of the profile image.
 *
 * @param  string|null  $image  The image name or null.
 * @return string               The URL of the profile image.
 */
if (!function_exists('profile_image_url')) {
	function profile_image_url($image)
	{
		if (!empty($image)) {
			$image = asset('images/profile/' . $image);
		} else {
			$image = asset('images/profile-placeholder.jpg');
		}

		return $image;
	}
}

/**
 * Get a human-readable date representation.
 *
 * @param  string  $date  The date string.
 * @return string         A human-readable date representation.
 */
if (!function_exists('readable_date')) {
	function readable_date($date)
	{
		$date = Carbon::parse($date);
		$readable_date = $date->diffForHumans( );

		return $readable_date;
	}
}

/**
 * Generate a unique image name.
 *
 * @param  string  $extension  The file extension.
 * @return string             The unique image name.
 */
if (!function_exists('unique_image_name')) {
	function unique_image_name($extension)
	{
		$name = date('Ymd') . time() . rand() . "." . $extension;
		return $name;
	}
}

/**
 * Generate an attendance status badge.
 *
 * @param  string  $status  The attendance status ('present' or 'absent').
 * @return string          The HTML badge representing the attendance status.
 */
if (!function_exists('attendance_status_badge')) {
	function attendance_status_badge($status)
	{
		if ($status == 'present') {
			return '<span class="badge badge-success"">Present</span>';
		} else {
			return '<span class="badge badge-primary">Absent</span>';
		}
	}
}

/**
 * Get the day name for a given date.
 *
 * @param  string|null  $date  The date in 'Y-m-d' format.
 * @return string              The abbreviated day name (e.g., Mon, Tue, etc.).
 */
if (!function_exists('day_name')) {
	function day_name($d)
	{
		$day_name = '';

		if (empty($d)) {
			return $day_name;
		}

		$date = Carbon::createFromFormat('Y-m-d', $d);
		$day_name = $date->format('l');

		return $day_name;
	}
}

/**
 * Get the start and end dates of the week for a given date.
 *
 * @param  string  $date  The date in 'Y-m-d' format.
 * @return array          An array containing the start and end dates of the week.
 */
if (!function_exists('week_dates')) {
	function week_dates($date)
	{
		$dates = ['date1' => '', 'date2' => ''];

		$date = Carbon::createFromFormat('Y-m-d', $date);
		$date->startOfWeek(Carbon::SUNDAY); // START OF THE WEEK IS SUNDAY

		$dates['date1'] = $date->copy();
		$dates['date2'] = $date->copy()->endOfWeek(Carbon::SATURDAY);
		
		return $dates;
	}
}

/**
 * Get the full path to an image located in the 'images/profile' directory.
 *
 * @param  string  $image  The name of the image file.
 * @return string          The full path to the image.
 */
if(!function_exists('get_image_path')) {
	function get_image_path($image) {
		$path = asset('images/profile/'.$image);
		return $path;
	}
}

if (!function_exists('attendance_status')) {
	function attendance_status($attendances)
	{	
		if(count($attendances) > 0) {
			foreach ($attendances as $attendance) {
				if ($attendance->status == 'present') {
					return '<span class="badge badge-success"">Present</span>';
				} else if ($attendance->status == 'absent') {
					return '<span class="badge badge-warning"">Absent</span>';
				} else {
					return '<span class="badge badge-secondary"">Not Taken</span>';
				}
			}
		} else {
			return '<span class="badge badge-info"">Not Taken</span>';
		}
	}
}